#!/bin/sh

szMachine=`uname -m | grep arm`

if [ -z $szMachine ]; then
	szMachine=x86
else
	szMachine=arm
fi

echo -en "GIGABYTE BMC Default Loader V1.00\n\n"
echo "Usage:"
echo "    ./BMCDefault.sh <BMC IP Address> LoadDefault"

echo -en "\n\nChecking default.conf ... "
if [ ! -e default.conf ]; then
	echo "NG"
	exit 1
fi

echo "OK"

strIPAddr=$1
strUsername=admin
strPassword=password
IANA="0A 3C 00"

Ipmitool()
{
	ipmitool -H $strIPAddr -U $strUsername -P $strPassword -I lanplus $*
	u8Retval=$?
	u8Retry=0

	while [ $u8Retval -ne 0 ]
	do
		ipmitool -H $strIPAddr -U $strUsername -P $strPassword -I lanplus $*
		u8Retval=$?

		let 'u8Retry++'
		if [ $u8Retry -eq 5 ]; then
			exit 1;
		fi
		sleep 2
	done

}

IpmiCmd()
{
	NetFn=$1
	CmdID=$2
	shift 2
	RawData="$*"

	IPMICmd 20 $NetFn 00 $CmdID $RawData
}

ExecCmd()
{
	if [ "$szMachine" = "x86" ]; then
		au8Hex=""

		for data in $*
		do
			au8Hex="$au8Hex 0x$data"
		done 
		Ipmitool raw $au8Hex
	else
		IpmiCmd $*
	fi
}

GetValue()
{
	cat default.conf | grep "$1" | awk '{ print $3 }'
}

InetPtoN()
{
	IPData1=`echo $1 | cut -d. -f 1 | xargs printf "%02x"`
	IPData2=`echo $1 | cut -d. -f 2 | xargs printf "%02x"`
	IPData3=`echo $1 | cut -d. -f 3 | xargs printf "%02x"`
	IPData4=`echo $1 | cut -d. -f 4 | xargs printf "%02x"`

	echo "$IPData1 $IPData2 $IPData3 $IPData4"
}

StrtoHex()
{
	echo -en "$1" | hexdump -v -e '1/1 " %02x"'
}

StrLen()
{
	echo -en "$1" | wc -c
}

# $1: String
# $2: the length of Padding with Zero
StrtoHexWithZeroPadding()
{
	u8ZeroLen=$2
	au8HexData=`echo -en "$1" | hexdump -v -e '1/1 " %02x"'`

	u8Len=`StrLen $1`
	u8Len=$((u8ZeroLen - u8Len))

	for i in `seq 1 $u8Len`
	do
		au8HexData="$au8HexData 0"
	done

	echo $au8HexData
}

# $1: Decimal digital
# $2: Digital number // byte, word, dword
# Return: Hex digital
Dec2Hex()
{
	Digital=$1

	case "$2" in
		1)
			printf "%x" $Digital
			;;
		2)
			printf "%x %x" $((Digital & 0x00ff)) $(((Digital & 0xff00) >> 8))
			;;
		4)
			printf "%x %x %x %x" \
					$((Digital & 0x0000ff)) $(((Digital & 0x0000ff00) >> 8)) \
					$(((Digital & 0xff0000) >> 16)) $((((Digital & 0xff000000)) >> 24))
			;;
	esac
}

# $1: string
# $2: seconds
PrintProgressBar ()
{
	for i in `seq 1 $2`
	do
		sleep 1
		echo -en "$1\033[1C"
		printf "%02d" $i
		echo -en "\r"
	done

	if [ "$3" != "" ]; then
		echo -en "$1\033[1C $3\n"
	else
		echo ""
	fi
}

fnLoadDomainHostname()
{
	u8Hostname=`GetValue "sz64_NET_HostName ="`
	u8Hostname=`StrtoHexWithZeroPadding "$u8Hostname" 64`

	u8Domain=`GetValue "sz64_NET_Domain ="`
	u8Domain=`StrtoHexWithZeroPadding "$u8Domain" 64`

	u8NetMode=`GetValue "u8___NET_Mode ="`
	bDns=`GetValue "b__NET_Global_DNS ="`
	if [ $bDns = "Enabled" ]; then
		bDns=1
	else
		bDns=0
	fi

	bDdns=`GetValue "lt__NET_Global_DDNS ="`
	if [ $bDdns = "Enabled" ]; then
		bDdns=1
	elif [ $bDdns = "Disabled" ]; then
		bDdns=0
	else
		# By Interface
		bDdns=2
	fi

	u8Raw="$u8Hostname $u8Domain $u8NetMode $bDns $bDdns"
	ExecCmd 2e 50 $IANA 01 $u8Raw 1> /dev/null

	PrintProgressBar "Load BMC Hostname & Domain ... " 10 "OK"
}

fnLoadNetServiceConfig()
{
	u16HttpPN=`GetValue "u16_SERVICES_HttpPortNumber ="`
	u16HttpsPN=`GetValue "u16_SERVICES_HttpsPortNumber ="`
	u16Timeout=`GetValue "u16_SERVICES_Timeout ="`

	u16HttpPN=`Dec2Hex $u16HttpPN 2`
	u16HttpsPN=`Dec2Hex $u16HttpsPN 2`
	u16Timeout=`Dec2Hex $u16Timeout 2`

	ExecCmd 2e 50 $IANA 04 \
	$u16Timeout $u16HttpPN $u16HttpsPN
}

fnLoadNetSecurityConfig()
{
	bIPBlockingEnabled=`GetValue       "b___NETSEC_IPBlockingEnabled ="`
	u8IPBlockingFailCount=`GetValue    "u8__NETSEC_IPBlockingFailCount ="`
	u16IPBlockingFailWindow=`GetValue  "u16_NETSEC_IPBlockingFailWindow ="`
	u16IPBlockingPenaltyTime=`GetValue "u16_NETSEC_IPBlockingPenaltyTime ="`

	if [ $bIPBlockingEnabled = "Enabled" ]; then
		bIPBlockingEnabled=1
	else
		bIPBlockingEnabled=0
	fi

	u8IPBlockingFailCount=`Dec2Hex $u8IPBlockingFailCount 1`
	u16IPBlockingFailWindow=`Dec2Hex $u16IPBlockingFailWindow 2`
	u16IPBlockingPenaltyTime=`Dec2Hex $u16IPBlockingPenaltyTime 2`

	ExecCmd 2e 50 $IANA 03 \
	"$bIPBlockingEnabled $u8IPBlockingFailCount $u16IPBlockingFailWindow $u16IPBlockingPenaltyTime"
}

fnLoadNTPConfig()
{
	u8NTPOpmode=`GetValue "lt_NTP_Operation_Mode ="`
	if [ "$u8NTPOpmode" = "Daemon" ]; then
		u8NTPOpmode="01"
	elif [ "$u8NTPOpmode" = "Requested" ]; then
		u8NTPOpmode="02"
	else
		u8NTPOpmode="00"
	fi

	u8NTPStepMode=`GetValue "b__NTP_Step_Mode ="`
	if [ "$u8NTPStepMode" = "Yes" ]; then
		u8NTPStepMode="00"
	else
		u8NTPStepMode="01"
	fi

	u8NTPUpdateFreq=`GetValue "u16__NTP_Update_Frequency ="`
	if [ $u8NTPUpdateFreq -lt 256 ]; then
		u8NTPUpdateFreq="$u8NTPUpdateFreq 0"
	else
		u8NTPUpdateFreq="$((u8NTPUpdateFreq % 256)) $((u8NTPUpdateFreq / 256))"
	fi

	au8NTPServer=`GetValue "sz48_NTP_Server_IP_Address ="`
	au8NTPServer=`InetPtoN $au8NTPServer`

	au8Raw="$u8NTPOpmode $u8NTPStepMode $u8NTPUpdateFreq $au8NTPServer"

	ExecCmd 2e 50 $IANA 0 $au8Raw

	echo "Load NTP Config ... OK"
}

fnLoadSNMPConfig()
{
	u8AlertACK=`GetValue "b__SNMP_Alert_ACK ="`
	if [ "$u8AlertACK" = "Yes" ]; then
		u8AlertACK="80"
	else
		u8AlertACK="00"
	fi

	u8AlertACKTimeout=`GetValue "u8_SNMP_Alert_ACK_Timeout ="`
	u8AlertACKTimeout=$((u8AlertACKTimeout & 0xff))

	u8AlertACKRetries=`GetValue "u8_SNMP_Alert_ACK_Retries ="`
	u8AlertACKRetries=$((u8AlertACKRetries & 0x07))

	for i in `seq 1 4`
	do
		u8AlertServer=`GetValue "sz48_SNMP_Trap_IP_Address$i ="`
		ExecCmd 0C 01 01 12 $i $u8AlertACK $u8AlertACKTimeout $u8AlertACKRetries > /dev/null
		ExecCmd 0C 01 01 13 $i 0 0 `InetPtoN $u8AlertServer` 0 0 0 0 0 0 > /dev/null
	done

	for i in `seq 1 4`
	do
		u8PEFPolicyEnabled=`GetValue "b__SNMP_PEF_Policy_Entry$i ="`
		if [ "$u8PEFPolicyEnabled" = "Enable" ]; then
			ExecCmd 04 12 9 $i 18 1$i 0 > /dev/null
		else
			ExecCmd 04 12 9 $i 10 1$i 0 > /dev/null
		fi
	done

#	for i in `seq 1 4`
#	do
#		Ipmitool raw 0x0C 0x02 0x01 18 $i 0
#		Ipmitool raw 0x0C 0x02 0x01 19 $i 0
#	done

	echo "Load SNMP Config ... OK"
}

fnLoadLDAPConfig()
{
	echo "Load LDAP Config ... OK"

	bLDAP_Enabled=`GetValue "b__LDAP_Enable ="`
	bLDAP_Enabled_Encryption=`GetValue "b__LDAP_Enable_Encryption ="`
	strLDAP_DNS_Search_Domain=`GetValue "sz64_LDAP_Domain_Name_for_DNS_SVR_request ="`
	strLDAP_DC1=`GetValue "sz48_LDAP_Domain_Controller1_IP ="`
	strLDAP_DC2=`GetValue "sz48_LDAP_Domain_Controller2_IP ="`
	strLDAP_DC3=`GetValue "sz48_LDAP_Domain_Controller3_IP ="`
	strLDAP_Base_DN=`GetValue "sz64_LDAP_Base_Domain_Name ="`
	strLDAP_Client_ID=`GetValue "sz64_LDAP_Client_ID ="`
	strLDAP_Client_Password=`GetValue "sz64_LDAP_Client_Password ="`

	if [ "$strLDAP_DNS_Search_Domain" = "" ]; then
		echo "Please assign Domain Name which can be found by DNS."
		exit 1
	fi

	if [ "$strLDAP_Base_DN" = "" ]; then
		echo "Base Domain Name not assigned."
		exit 1
	fi

	if [ "$strLDAP_Client_ID" = "" ]; then
		echo "Client ID not assigned."
		exit 1
	fi

	if [ "$strLDAP_Client_Password" = "" ]; then
		echo "Client Password not assigned."
		exit 1
	fi

	# Configuration 1
	au8Raw=""
	if [ "$bLDAP_Enabled" = "Yes" ]; then
		au8Raw="01"
	else
		au8Raw="00"
	fi

	if [ "$bLDAP_Enabled_Encryption" = "Yes" ]; then
		au8Raw="$au8Raw 01"
	else
		au8Raw="$au8Raw 00"
	fi

	au8Raw="$au8Raw `StrtoHexWithZeroPadding "$strLDAP_DNS_Search_Domain" 64`"
	au8Raw="$au8Raw `StrtoHex "$strLDAP_Base_DN"`"

	ExecCmd 2e 50 $IANA 2 1 $au8Raw 0

	# Configuration 2
	au8Raw=""
	au8Raw="`StrtoHexWithZeroPadding "$strLDAP_Client_ID" 64`"
	au8Raw="$au8Raw `StrtoHex "$strLDAP_Client_Password"`"

	ExecCmd 2e 50 $IANA 2 2 $au8Raw 0

	# Configuration 3
	au8Raw=""
	au8Raw="`StrtoHexWithZeroPadding "$strLDAP_DC1" 64`"
	au8Raw="$au8Raw `StrtoHexWithZeroPadding "$strLDAP_DC2" 64`"
	au8Raw="$au8Raw `StrtoHex "$strLDAP_DC3"`"

	ExecCmd 2e 50 $IANA 2 3 $au8Raw 0
}

fnLoadIPMIConfig()
{
	au8SetSerialCmd="0C 10 02"

	bSerialEnabled=`GetValue "b__IPMI_SerialEnabled ="`
	strSerialConnMode=`GetValue "lt_IPMI_SerialConnectionMode ="`
	strSerialBaudRate=`GetValue "lt_IPMI_SerialBaudRate ="`
	strSerialFlowControl=`GetValue "lt_IPMI_SerialFlowControl ="`
	strSerialChannelPLL=`GetValue "lt_IPMI_SerialChannelPLLimit ="`

	bLanEnabled=`GetValue "b__IPMI_LANEnabled ="`
	strLanChannelPLL=`GetValue "lt_IPMI_LANChannelPLLimit ="`
	strLanSecurityKeys="`GetValue "au20_IPMI_LANSecurityKeys ="`"

	#
	# Serial Channel Enabled/Disabled
	#
	if [ $bSerialEnabled = "Enabled" ]; then
		bSerialEnabled=ba
		bSerialEnabledNonVol=7a
	else
		bSerialEnabled=b8
		bSerialEnabledNonVol=78
	fi

	case "$strSerialChannelPLL" in
		User)
			strSerialChannelPLL=82
			strSerialChannelPLLNonVol=42
			;;

		Operator)
			strSerialChannelPLL=83
			strSerialChannelPLLNonVol=43
			;;

		Administrator)
			strSerialChannelPLL=84
			strSerialChannelPLLNonVol=44
			;;

		*)
			strSerialChannelPLL=84
			strSerialChannelPLLNonVol=44
			;;
	esac

	ExecCmd 06 40 2 $bSerialEnabledNonVol $strSerialChannelPLLNonVol
	ExecCmd 06 40 2 $bSerialEnabled $strSerialChannelPLL

	#
	# Parameter 3: Connection Mode
	#
	if [ "$strSerialConnMode" = "Basic" ]; then
		strSerialConnMode=81
	else
		strSerialConnMode=84
	fi

	ExecCmd $au8SetSerialCmd 3 $strSerialConnMode

	#
	# Parameter 7: IPMI Messaging Comm Settings
	#
	case "$strSerialBaudRate" in
		9600)
			strSerialBaudRate=06
			;;
		19.2)
			strSerialBaudRate=07
			;;
		38.4)
			strSerialBaudRate=08
			;;
		57.6)
			strSerialBaudRate=09
			;;
		115.2)
			strSerialBaudRate=0A
			;;
		*)
			strSerialBaudRate=09
			;;
	esac

	if [ "$strSerialFlowControl" = "None" ]; then
		strSerialFlowControl=0
	else
		strSerialFlowControl=40
	fi

	ExecCmd $au8SetSerialCmd 7 $strSerialFlowControl $strSerialBaudRate

	#
	# LAN Channel Enable/Disable
	#
	if [ $bLanEnabled = "Enabled" ]; then
		bLanEnabled=82
		bLanEnabledNonVol=42
	else
		bLanEnabled=80
		bLanEnabledNonVol=40
	fi

	case "$strLanChannelPLL" in
		User)
			strLanChannelPLL=82
			strLanChannelPLLNV=42
			;;

		Operator)
			strLanChannelPLL=83
			strLanChannelPLLNV=43
			;;

		Administrator)
			strLanChannelPLL=84
			strLanChannelPLLNV=44
			;;
		*)
			strLanChannelPLL=84
			strLanChannelPLLNV=44
			;;
	esac

	ExecCmd 06 40 1 $bLanEnabledNonVol $strLanChannelPLLNV
	ExecCmd 06 40 1 $bLanEnabled $strLanChannelPLL

	#
	# Set Lan Channel Security Key
	#
	# TODO: Not Implement yet!
	return 0
	strKeyValue=""
	for i in $strLanSecurityKeys
	do
		strKeyValue="$strKeyValue $i"
	done
	ExecCmd 06 56 1 1 1 $strKeyValue
}

fnLoadLanguage()
{
	strLanguage=`GetValue "lt__GUI_Language ="`
	strLanguage=`StrtoHexWithZeroPadding $strLanguage 8`
	ExecCmd 2e 50 $IANA 5 $strLanguage
}

fnResetToDefault()
{
	ExecCmd 2e cc 5e 2b 00 0a 02 aa > /dev/null
	PrintProgressBar "Reset BMC to Factory Default, please wait 60 seconds: " 60

	echo "Done"
}

fnLoadAll()
{
	fnLoadNTPConfig
	fnLoadSNMPConfig
	fnLoadLDAPConfig
	fnLoadNetSecurityConfig
	fnLoadNetServiceConfig
	fnLoadIPMIConfig
	fnLoadLanguage
	fnLoadDomainHostname
}

case $2 in
	Domain)
		fnLoadDomainHostname
		;;

	Hostname)
		fnLoadDomainHostname
		;;

	NetSecu)
		fnLoadNetSecurityConfig
		;;

	NetSvc)
		fnLoadNetServiceConfig
		;;

	IPMI)
		fnLoadIPMIConfig
		;;

	NTP)
		fnLoadNTPConfig
		;;

	LANG)
		fnLoadLanguage
		;;

	SNMP)
		fnLoadSNMPConfig
		;;

	LDAP)
		fnLoadLDAPConfig
		;;

	Reset)
		fnResetToDefault
		;;

	LoadDefault)
		fnLoadAll
		;;

	Test)
		Dec2Hex 343 2
		echo ""
		;;
esac

